/*
 
 File: GenesysMobileService.java
 Abstract: Encapsulates a service provide by Genesys Mobile Services
 
 Version: 1.0
 
 Disclaimer: IMPORTANT:  This software is supplied to you by Genesys
 Telecommunications Laboratories Inc ("Genesys") in consideration of your agreement
 to the following terms, and your use, installation, modification or redistribution
 of this Genesys software constitutes acceptance of these terms.  If you do not
 agree with these terms, please do not use, install, modify or redistribute this
 Genesys software.
 
 In consideration of your agreement to abide by the following terms, and subject
 to these terms, Genesys grants you a personal, non-exclusive license, under
 Genesys's copyrights in this original Genesys software (the "Genesys Software"), to
 use, reproduce, modify and redistribute the Genesys Software, with or without
 modifications, in source and/or binary forms; provided that if you redistribute
 the Genesys Software in its entirety and without modifications, you must retain
 this notice and the following text and disclaimers in all such redistributions
 of the Genesys Software.
 
 Neither the name, trademarks, service marks or logos of Genesys Inc. may be used
 to endorse or promote products derived from the Genesys Software without specific
 prior written permission from Genesys.  Except as expressly stated in this notice,
 no other rights or licenses, express or implied, are granted by Genesys herein,
 including but not limited to any patent rights that may be infringed by your
 derivative works or by other works in which the Genesys Software may be
 incorporated.
 
 The Genesys Software is provided by Genesys on an "AS IS" basis.  GENESYS MAKES NO
 WARRANTIES, EXPRESS OR IMPLIED, INCLUDING WITHOUT LIMITATION THE IMPLIED
 WARRANTIES OF NON-INFRINGEMENT, MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 PURPOSE, REGARDING THE GENESYS SOFTWARE OR ITS USE AND OPERATION ALONE OR IN
 COMBINATION WITH YOUR PRODUCTS.
 
 IN NO EVENT SHALL GENESYS BE LIABLE FOR ANY SPECIAL, INDIRECT, INCIDENTAL OR
 CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
 GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 ARISING IN ANY WAY OUT OF THE USE, REPRODUCTION, MODIFICATION AND/OR
 DISTRIBUTION OF THE GENESYS SOFTWARE, HOWEVER CAUSED AND WHETHER UNDER THEORY OF
 CONTRACT, TORT (INCLUDING NEGLIGENCE), STRICT LIABILITY OR OTHERWISE, EVEN IF
 GENESYS HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 
 Copyright (C) 2012 Genesys Inc. All Rights Reserved.
 
 */

package com.genesyslab.gms.android.framework;

import java.io.BufferedReader;
import java.io.ByteArrayOutputStream;
import java.io.InputStreamReader;
import java.net.HttpURLConnection;
import java.net.URL;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.Observable;
import java.util.Observer;


import android.os.AsyncTask;
import android.os.Bundle;
import android.util.Log;

public class GenesysMobileService implements Observer {
	
	public static final int REQUEST_CREATE = 0;
	public static final int REQUEST_ACTION = 1;
	public static final int REQUEST_NOTIFY = 2;

	private static final String base64code = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";

	private String gmsHost;
	private String gmsUserName;
	private String gmsPassword; 
	private boolean isDebug; //Enable verbose logging
	private ArrayList<GenesysMobileServiceListener> _listeners = new ArrayList<GenesysMobileServiceListener>(); //Listeners to receive callbacks
	private int currentRequest;
	
	/**
	 * @param gmsHost The base URL of Genesys Mobile Services
	 * @param gmsUserName The username if any
	 * @param gmsPassword The password if any
	 */
	public GenesysMobileService(String gmsHost, String gmsUserName,
			String gmsPassword) {
		super();
		this.gmsHost = gmsHost;
		this.gmsUserName = gmsUserName;
		this.gmsPassword = gmsPassword;
	}

	//Getters and Setters
	
	public String getGmsHost() {
		return gmsHost;
	}

	public void setGmsHost(String gmsHost) {
		this.gmsHost = gmsHost;
	}

	public String getGmsUserName() {
		return gmsUserName;
	}

	public void setGmsUserName(String gmsUserName) {
		this.gmsUserName = gmsUserName;
	}

	public String getGmsPassword() {
		return gmsPassword;
	}

	public void setGmsPassword(String gmsPassword) {
		this.gmsPassword = gmsPassword;
	}

	public boolean isDebug() {
		return isDebug;
	}

	public void setDebug(boolean isDebug) {
		this.isDebug = isDebug;
	}

	public synchronized void addEventListener(GenesysMobileServiceListener listener)  {
		    _listeners.add(listener);
	}

	public synchronized void removeEventListener(GenesysMobileServiceListener listener)   {
		    _listeners.remove(listener);
	}
	
	// call this method whenever you want to notify
	  //the event listeners of the particular event
	  private synchronized void fireEvent(GMSEvent event) {
	    Iterator<GenesysMobileServiceListener> i = _listeners.iterator();
	    while(i.hasNext())  {
	      GenesysMobileServiceListener listener = ((GenesysMobileServiceListener) i.next());
	      if (event.isError())
	    	  listener.onFailWithError(event);
	      else if (this.currentRequest == REQUEST_CREATE) {
	    	  listener.onCreateGenesysMobileService(event);
	      }
	      else if (this.currentRequest == REQUEST_ACTION) {
	    	  listener.onActionPerformed(event);
	      }
	      else {
	    	  listener.onServerEvent(event);
	      }
	    }
	  }
	
	/**
	 * Create a Genesys Mobile Service request. You can upload any custom data
	 * that is sent as a http multipart/formdata post
	 */
	public void createService(GenesysMobileServiceRequest gmsServiceRequest, String serviceAlias) {
	    currentRequest = REQUEST_CREATE;
	    try {
			URL url = new URL(this.gmsHost + serviceAlias);
			new HTTPTask().execute(url, gmsServiceRequest);
	    }
	    catch(Exception ex){
		    GMSEvent event = new GMSEvent(this);
		    event.setError(true);
		    event.setGmsResponse(ex.toString());
		    this.fireEvent(event);
	    }
	}

	/**
	 * Perform action on a Genesys Mobile Session. You can upload any custom data
	 * as a http multipart/formdata post
	 */
	public void performAction(String action, String genesysMobileServiceSessionId, GenesysMobileServiceRequest gmsServiceRequest) {
	    currentRequest = REQUEST_ACTION;
	    try {
			URL url = new URL(this.gmsHost + genesysMobileServiceSessionId + "/" + action);
			new HTTPTask().execute(url, gmsServiceRequest);
	    }
	    catch(Exception ex){
		    GMSEvent event = new GMSEvent(this);
		    event.setError(true);
		    event.setGmsResponse(ex.toString());
		    this.fireEvent(event);
	    }
	}
	
	private class HTTPTask extends AsyncTask<Object, Void, GMSEvent> {

		@Override
		protected GMSEvent doInBackground(Object... params) {
			URL url = (URL) params[0];
			GenesysMobileServiceRequest gmsServiceRequest = (GenesysMobileServiceRequest) params[1];

			try {
				HttpURLConnection urlConnection = (HttpURLConnection) url.openConnection();
		    	urlConnection.setDoOutput(true);
		    	urlConnection.setReadTimeout(60000);
		    	urlConnection.setRequestMethod("POST");
		    	urlConnection.setRequestProperty("Content-Type", "multipart/form-data; boundary=AaB03x");
		    	urlConnection.setRequestProperty("Accept", "application/json");
		    	
		    	if ((gmsUserName != null) && (gmsPassword != null)) {
		    		String credentials = GenesysMobileService.base64Encode(gmsUserName + ":" + gmsPassword); 
		    		if (isDebug) {
		    			Log.d("credentials", credentials);
		    		}
		    		urlConnection.setRequestProperty("Authorization", "Basic " + credentials);
		    	}
		    	
		        // send data
		    	ByteArrayOutputStream bos = gmsServiceRequest.getRequestBodyAsStream();
		    	bos.writeTo(urlConnection.getOutputStream());
			    
			    // evaluate response code
			    // if we were successful just return the response
			    // if unsuccessful return response message
			    int response_code = urlConnection.getResponseCode();
			    if(response_code != 200) {
			    	String response = urlConnection.getResponseMessage();

			    	if (isDebug) {
			    		BufferedReader rd = new BufferedReader(new InputStreamReader(urlConnection.getInputStream()));
			    		String line;
			    		StringBuffer buffer = new StringBuffer();
			    		while ((line = rd.readLine()) != null) {
			    			buffer.append(line);
			    		}
			    		Log.d("GMS Server Error response: ", buffer.toString());
			    	}

				    GMSEvent event = new GMSEvent(this);
				    event.setError(true);
				    event.setGmsResponse(response);
				    return event;
			    }
			    else {
				    BufferedReader rd = new BufferedReader(new InputStreamReader(urlConnection.getInputStream()));
				    String line;
				    StringBuffer buffer = new StringBuffer();
				    while ((line = rd.readLine()) != null) {
				    	buffer.append(line);
				    }
				    
				    String response = buffer.toString();		    	
				    GMSEvent event = new GMSEvent(this);
				    event.setError(false);
				    event.setGmsResponse(response);
				    return event;
			    }
			}
		    catch(Exception ex){
			    GMSEvent event = new GMSEvent(this);
			    event.setError(true);
			    event.setGmsResponse(ex.toString());
			    return event;
		    }
		}
		
    	protected void onPostExecute(GMSEvent gmsEvent) {   
    		fireEvent(gmsEvent);
    	}
	}
	
	private static byte[] zeroPad(int length, byte[] bytes) {
        byte[] padded = new byte[length]; // initialized to zero by JVM
        System.arraycopy(bytes, 0, padded, 0, bytes.length);
        return padded;
    }
 
    public static String base64Encode(String string) {
 
        String encoded = "";
        byte[] stringArray;
        try {
            stringArray = string.getBytes("UTF-8");  // use appropriate encoding string!
        } catch (Exception ignored) {
            stringArray = string.getBytes();  // use locale default rather than croak
        }
        // determine how many padding bytes to add to the output
        int paddingCount = (3 - (stringArray.length % 3)) % 3;
        // add any necessary padding to the input
        stringArray = zeroPad(stringArray.length + paddingCount, stringArray);
        // process 3 bytes at a time, churning out 4 output bytes
        // worry about CRLF insertions later
        for (int i = 0; i < stringArray.length; i += 3) {
            int j = ((stringArray[i] & 0xff) << 16) +
                ((stringArray[i + 1] & 0xff) << 8) + 
                (stringArray[i + 2] & 0xff);
            encoded = encoded + base64code.charAt((j >> 18) & 0x3f) +
                base64code.charAt((j >> 12) & 0x3f) +
                base64code.charAt((j >> 6) & 0x3f) +
                base64code.charAt(j & 0x3f);
        }
        // replace encoded padding nulls with "="
        return encoded.substring(0, encoded.length() - paddingCount) + "==".substring(0, paddingCount);
 
    }
    
    /*
     * Handles notifications from C2DMReceiver
     * @see java.util.Observer#update(java.util.Observable, java.lang.Object)
     */

	public void update(Observable arg0, Object arg1) {
		if (arg1 instanceof Bundle) {
			Bundle bundle = (Bundle) arg1;
			String type = bundle.getString("type");
			if(type.contains("c2dmmessage")){
				this.currentRequest = REQUEST_NOTIFY;
				String message = bundle.getString("message");
			    GMSEvent event = new GMSEvent(this);
			    event.setError(false);
			    event.setGmsResponse(message);
			    this.fireEvent(event);
			}
		}
	}

}
